<?php

namespace App\Http\Controllers\Auth;

use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class UserAuthController extends Controller
{
    public function registerPage()
    {
        return view('users.register');
    }

    public function loginPage()
    {
        return view('users.login');
    }

    public function forgotPage()
    {
        return view('users.forgot-password');
    }

    public function resetPage($token)
    {
        return view('users.reset-password', compact('token'));
    }

    public function registerPost(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email',
            'password' => 'required|min:8',
        ]);

        if ($validator->fails()) {
            // Validation failed, redirect back with errors and input data
            return redirect()->back()->with('info', 'Please enter all fields data !');
        }
        // dd($request->all());
        // Validation passed, proceed with user registration
        $user = new User();
        $user->name = $request->name;
        $user->email = $request->email;
        $user->join_date = Carbon::now()->format('F d, Y');
        $user->password = Hash::make($request->password);
        if ($request->hasFile('profile')) {
            $file = $request->file('profile');
            $fileName = time() . '.' . $file->getClientOriginalExtension();
            $path = $file->move('uploads/users/', $fileName);
            $user->profile = $fileName;
        }
        $user->save();
        return redirect()->route('login.get')->with('success', 'Registered user successfully !');
    }

    public function loginPost(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required | email | exists:users',
            'password' => 'required'
        ]);
        // dd($request->all());
        if (Auth::attempt($credentials)) {
            if (Auth::user()->role == 1) {
                return redirect()->route('admin.index.get')->with('success', 'Admin login successfully !!!');
            } else {
                return redirect()->route('index.get')->with('success', 'User login successfully !!!');
            }
        }
        return back()->with('error', 'Record not matched with data !!!');
    }

    public function forgotPost(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users',
        ]);

        $token = mt_rand(100000, 999999);

        DB::table('password_resets')->insert([
            'email' => $request->email,
            'token' => $token,
            'created_at' => Carbon::now()
        ]);

        Mail::send('emails.forgot-password', ['token' => $token], function ($message) use ($request) {
            $message->to($request->email);
            $message->subject('Reset Password Link');
        });
        return back()->with('success', 'Successfully send the reset password link on your email please check to verify !!!');
    }

    public function resetPost(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required | email | exists:users',
            'password' => 'required | min:8 | same:password',
            'cpassword' => 'required'
        ]);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }
        $updatePassword = DB::table('password_resets')
            ->where([
                'email' => $request->email,
                'token' => $request->token
            ])->first();
            // dd($updatePassword);
        if (!$updatePassword) {
            return back()->with('error', 'Something went wrong , password not updated !!!');
        }
        $user = User::where('email', $request->email)
            ->update(['password' => Hash::make($request->password)]);
        DB::table('password_resets')->where(['email' => $request->email])->delete();
        return redirect()->route('index.get')->with('success', 'Successfully update password !!!');
    }

    public function logout()
    {
        Auth::logout();
        return redirect()->route('index.get')->with('success','Logout Successfully !!!');
    }
}
