<?php

namespace App\Http\Controllers\Auth;

use App\Profile;
use App\User;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Auth\Events\Registered;
use Session;

use DB;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = '/'; 

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => 'required|string',
            'email' => 'required|string|email|unique:users',
            'password' => 'required|min:8',
        ]);
    }

    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request)
    {
        $validator = $this->validator($request->all());
        if($validator->fails()){
            return redirect()->back()->withInput()->withErrors($validator, 'registerForm');
        }

        event(new Registered($user = $this->create($request->all())));

        $this->guard()->login($user);
        
        Session::flash('message', 'New Account Created Successfully'); 
        Session::flash('alert-class', 'alert-success');
        return $this->registered($request, $user)
                        ?: redirect($this->redirectPath());
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\User
     */
    protected function create(array $data)
    {
        // dd('ok');
        return User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
        ]);
    }

    protected function registered(Request $request, $user)
    {
        if($user->profile == null){
            $profile = new Profile();
            $profile->user_id = $user->id;
            $profile->localisation = $request->localisation;
            $profile->dob = $request->dob;
            $profile->save();
        }
        activity($user->name)
            ->performedOn($user)
            ->causedBy($user)
            ->log('Registered');
        $user->assignRole('user');
    }
    
    public function registerPost(Request $request)
{
    $request->validate([
        'name' => 'required',
        'email' => 'required|email',
        'password' => 'required|min:8',
        'password_confirmation' => 'required|same:password',
        // 'profile' => 'image|mimes:jpeg,png,jpg,gif|max:2048', // Adjust the validation rules for the profile image
        'account_type' => 'required|in:user,creator', // Ensure account_type is either 'user' or 'creator'
    ]);

    $user = new User();
    $pic = new Profile();

    $user->name = $request->name;
    $user->email = $request->email;
    $user->password = Hash::make($request->password);

    if ($request->hasFile('profile')) {
        $file = $request->file('profile');
        $fileName = time() . '.' . $file->getClientOriginalExtension();
        $file->move('uploads/users/', $fileName);
        $pic->pic = $fileName;
        $user->profile = $fileName;
    }

    $user->save();

    // Determine role based on account_type
    $roleId = null;
    switch ($request->account_type) {
        case 'user':
            $roleId = DB::table('roles')->where('name', 'user')->value('id');
            break;
        case 'creator':
            $roleId = DB::table('roles')->where('name', 'creator')->value('id');
            break;
    }

    // Save role to role_user table
    if ($roleId) {
        DB::table('role_user')->insert([
            'role_id' => $roleId,
            'user_id' => $user->id,
        ]);
    }

    return redirect()->route('signin')->with('success', 'Registered user successfully !!!');
}


}
